<?php

if (!class_exists('SmartCrawl_Importer')) require_once 'class_wds_importer.php';

class SmartCrawl_Yoast_Importer extends SmartCrawl_Importer
{
	const IMPORT_IN_PROGRESS_FLAG = 'wds-yoast-import-in-progress';
	const NETWORK_IMPORT_SITES_PROCESSED_COUNT = 'wds-yoast-network-sites-processed';

	private $custom_handlers = array(
		'wpseo-premium-redirects-base'      => 'handle_redirects',
		'wpseo_titles/noindex-tax-TAXONOMY' => 'handle_taxonomy_noindex'
	);

	function data_exists()
	{
		$options = get_option('wpseo');
		$version = smartcrawl_get_array_value($options, 'version');

		if (!$version) {
			return false;
		}

		return strpos($version, '7.') === 0;
	}

	function import_options()
	{
		$mappings = $this->expand_mappings($this->load_option_mappings());
		$source_options = $this->get_yoast_options();
		$target_options = array();
		foreach ($source_options as $source_key => $source_value) {
			$target_key = smartcrawl_get_array_value($mappings, $source_key);

			if (!$target_key) {
				$target_options = $this->try_custom_handlers($source_key, $source_value, $target_options);
				continue;
			}

			$processed_target_key = $this->pre_process_key($target_key);
			$processed_target_value = $this->pre_process_value($target_key, $source_value);

			if (!$processed_target_key) {
				continue;
			}

			smartcrawl_put_array_value(
				$processed_target_value,
				$target_options,
				$processed_target_key
			);
		}

		$target_options = $this->activate_modules($target_options);
		$target_options = $this->activate_social_options_for_types($target_options);

		$this->save_options($target_options);
	}

	protected function get_pre_processors()
	{
		return array(
			'wds_onpage_options/preset-separator' => 'process_separator_value',
			'wds_social_options/schema_type'      => 'process_schema_type_value',
			'_wds_meta-robots-noindex'            => 'process_robots_noindex_value'
		);
	}

	protected function get_next_network_site_option_id()
	{
		return self::NETWORK_IMPORT_SITES_PROCESSED_COUNT;
	}

	protected function get_import_in_progress_option_id()
	{
		return self::IMPORT_IN_PROGRESS_FLAG;
	}

	protected function get_custom_handlers()
	{
		return $this->custom_handlers;
	}

	public function process_separator_value($target_key, $source_value)
	{
		$mapping = array(
			'sc-dash'   => 'dash',
			'sc-ndash'  => 'dash-l',
			'sc-mdash'  => 'dash-l',
			'sc-middot' => 'dot',
			'sc-bull'   => 'dot-l',
			'sc-star'   => 'pipe',
			'sc-smstar' => 'pipe',
			'sc-pipe'   => 'pipe',
			'sc-tilde'  => 'tilde',
			'sc-laquo'  => 'caret-left',
			'sc-raquo'  => 'caret-right',
			'sc-lt'     => 'less-than',
			'sc-gt'     => 'greater-than',
		);
		$mapped_value = smartcrawl_get_array_value($mapping, $source_value);

		if (!$mapped_value) {
			return $source_value;
		}

		return $mapped_value;
	}

	public function handle_taxonomy_noindex($source_key, $source_value, $target_options)
	{
		$current_taxonomy = null;
		foreach ($this->get_taxonomies() as $taxonomy) {
			if (str_replace('TAXONOMY', $taxonomy, 'wpseo_titles/noindex-tax-TAXONOMY') == $source_key) {
				$current_taxonomy = $taxonomy;
			}
		}

		if (!$current_taxonomy) {
			return $target_options;
		}

		$noindex_key = array('wds_onpage_options', sprintf('meta_robots-noindex-%s', $current_taxonomy));
		$sitemap_exclusion_key = array('wds_sitemap_options', sprintf('taxonomies-%s-not_in_sitemap', $current_taxonomy));

		smartcrawl_put_array_value($source_value, $target_options, $noindex_key);
		smartcrawl_put_array_value($source_value, $target_options, $sitemap_exclusion_key);
		return $target_options;
	}

	public function handle_redirects($source_key, $redirects, $target_options)
	{
		if (empty($redirects)) {
			return $target_options;
		}

		$wds_redirects = array();
		$wds_redirect_types = array();
		foreach ($redirects as $redirect) {
			$format = smartcrawl_get_array_value($redirect, 'format');
			$origin = smartcrawl_get_array_value($redirect, 'origin');
			$url = smartcrawl_get_array_value($redirect, 'url');
			$type = smartcrawl_get_array_value($redirect, 'type');

			// We are not supporting anything other than plain redirects at the moment
			if ($format != 'plain' || !$origin || !$url || !$type) {
				continue;
			}

			$wds_redirects[ home_url($origin) ] = home_url($url);
			$wds_redirect_types[ home_url($origin) ] = $type;
		}
		smartcrawl_put_array_value($wds_redirects, $target_options, 'wds-redirections');
		smartcrawl_put_array_value($wds_redirect_types, $target_options, 'wds-redirections-types');

		return $target_options;
	}

	public function process_robots_noindex_value($target_key, $source_value, $post_id)
	{
		$options = get_option('wpseo_titles');
		$noindex_setting = (boolean)smartcrawl_get_array_value(
			$options,
			sprintf('noindex-%s', get_post_type($post_id))
		);
		if (empty($source_value)) {
			$noindex_post = $noindex_setting;
		} else {
			$noindex_post = intval($source_value) === 1;
		}

		return $noindex_post;
	}

	public function process_schema_type_value($target_key, $source_value)
	{
		return $source_value == 'company' ? 'Organization' : 'Person';
	}

	function import_taxonomy_meta()
	{
		$mappings = $this->load_taxonomy_meta_mappings();
		$taxonomy_meta_option = get_option('wpseo_taxonomy_meta', array());
		$wds_meta = array();

		foreach ($taxonomy_meta_option as $taxonomy => $taxonomy_meta) {
			foreach ($taxonomy_meta as $term_id => $meta_values) {
				$wds_taxonomy_meta = array();

				foreach ($meta_values as $meta_key => $meta_value) {
					$target_key = smartcrawl_get_array_value($mappings, $meta_key);

					if (!$target_key || !$meta_value) {
						continue;
					}

					$processed_target_key = $this->pre_process_key($target_key);
					$processed_target_value = $this->pre_process_value($target_key, $meta_value);

					if (!$processed_target_key || !$processed_target_value) {
						continue;
					}

					smartcrawl_put_array_value(
						$processed_target_value,
						$wds_taxonomy_meta,
						$processed_target_key
					);
				}

				$wds_meta[ $taxonomy ][ $term_id ] = $wds_taxonomy_meta;
			}
		}
		update_option('wds_taxonomy_meta', $wds_meta);
	}

	function import_post_meta()
	{
		$mappings = $this->load_post_meta_mappings();
		$batch_size = apply_filters('wds_post_meta_import_batch_size', 300);
		$all_posts = $this->get_posts_with_yoast_metas();
		$batch_posts = array_slice($all_posts, 0, $batch_size);

		foreach ($batch_posts as $post_id) {
			$wds_meta = array();

			foreach ($mappings as $source_key => $target_key) {
				$target_value = get_post_meta($post_id, $source_key, true);

				if (!$target_key) {
					continue;
				}

				$processed_target_key = $this->pre_process_key($target_key);
				$processed_target_value = $this->pre_process_value($target_key, $target_value, $post_id);

				if (!$processed_target_key) {
					continue;
				}

				smartcrawl_put_array_value($processed_target_value, $wds_meta, $processed_target_key);
			}

			$this->add_post_meta($post_id, $wds_meta);
		}

		return count($all_posts) == count($batch_posts);
	}

	private function add_post_meta($post_id, $meta)
	{
		foreach ($meta as $meta_key => $meta_value) {
			update_post_meta($post_id, $meta_key, $meta_value);
		}
	}

	public function load_taxonomy_meta_mappings()
	{
		return $this->load_mapping_file('yoast-taxonomy-meta-mappings.php');
	}

	private function load_post_meta_mappings()
	{
		return $this->load_mapping_file('yoast-post-meta-mappings.php');
	}

	/**
	 * @return int[]
	 */
	private function get_posts_with_yoast_metas()
	{
		return $this->get_posts_with_source_metas('_yoast_');
	}

	private function load_option_mappings()
	{
		return $this->load_mapping_file('yoast-mappings.php');
	}

	private function get_yoast_options()
	{
		$all_options = array();
		$keys = array('wpseo', 'wpseo_titles', 'wpseo_social');

		foreach ($keys as $key) {
			$options = get_option($key);
			if (is_array($options) && count($options)) {
				foreach ($options as $option_key => $option_value) {
					$all_options[ $key . '/' . $option_key ] = $option_value;
				}
			}
		}
		return $this->add_redirects($all_options);
	}

	/**
	 * @param $all_options
	 * @return array
	 */
	private function add_redirects($all_options)
	{
		$option_id = 'wpseo-premium-redirects-base';
		$redirects = get_option($option_id);
		if ($redirects) {
			$all_options = array_merge(
				$all_options,
				array($option_id => $redirects)
			);
			return $all_options;
		}
		return $all_options;
	}

	private function activate_modules($target_options)
	{
		smartcrawl_put_array_value(true, $target_options, array('wds_settings_options', 'onpage'));
		smartcrawl_put_array_value(true, $target_options, array('wds_settings_options', 'social'));

		return $target_options;
	}

	private function get_all_supported_types()
	{
		return array_merge(
			array(
				'home', 'search',
				'404', 'bp_groups',
				'bp_profile', 'mp_marketplace-base',
				'mp_marketplace-categories', 'mp_marketplace-tags',
				'author', 'date',
			),
			$this->get_post_types(),
			$this->get_taxonomies()
		);
	}

	private function activate_social_options_for_types($target_options)
	{
		$og_enabled = smartcrawl_get_array_value($target_options, array('wds_social_options', 'og-enable'));
		$twitter_enabled = smartcrawl_get_array_value($target_options, array('wds_social_options', 'twitter-card-enable'));

		if ($og_enabled) {
			foreach ($this->get_all_supported_types() as $type) {
				smartcrawl_put_array_value(true, $target_options, array('wds_onpage_options', sprintf('og-active-%s', $type)));
			}
		}

		if ($twitter_enabled) {
			foreach ($this->get_all_supported_types() as $type) {
				smartcrawl_put_array_value(true, $target_options, array('wds_onpage_options', sprintf('twitter-active-%s', $type)));
			}
		}

		return $target_options;
	}
}